# International Payment Gateway System

A secure, dynamic payment gateway system with admin panel for creating payment links and accepting international card payments.

## 🔐 Security Features

- **End-to-end encryption** for sensitive data
- **CSRF protection** on all forms
- **Rate limiting** to prevent abuse
- **SQL injection protection** with prepared statements
- **XSS prevention** with input sanitization
- **Password hashing** with bcrypt
- **Session management** with secure tokens
- **PCI DSS compliant** (uses Stripe for card processing)
- **No card data storage** - only last 4 digits and brand
- **Encrypted configuration** - API keys encrypted at rest

## ✨ Features

### Admin Panel
- ✅ Secure login with session management
- ✅ Dashboard with transaction statistics
- ✅ Create dynamic payment links
- ✅ Manage gateway configurations
- ✅ View transaction history (no sensitive data exposed)
- ✅ Multi-currency support
- ✅ Recurring payment options
- ✅ Payment link expiration and usage limits
- ✅ Role-based access control

### Payment Processing
- ✅ Accept Visa, Mastercard, American Express, and more
- ✅ International payments support
- ✅ Real-time payment processing
- ✅ Custom redirect URLs
- ✅ Responsive payment page design
- ✅ Stripe integration (easily adaptable to other processors)

## 📋 Requirements

- PHP 7.4 or higher
- MySQL 5.7 or higher
- cURL extension enabled
- OpenSSL extension enabled
- PDO MySQL extension
- Web server (Apache/Nginx)
- Stripe account (or other payment processor)

## 🚀 Installation

### Step 1: Database Setup

1. Create a new MySQL database:
```sql
CREATE DATABASE payment_gateway CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

2. Import the database schema:
```bash
mysql -u your_username -p payment_gateway < database.sql
```

### Step 2: Configuration

1. Edit `config.php` and update the following:

```php
// Database credentials
define('DB_HOST', 'localhost');
define('DB_NAME', 'payment_gateway');
define('DB_USER', 'your_db_user');
define('DB_PASS', 'your_db_password');

// Generate a secure encryption key (32 characters)
// Run: php -r "echo base64_encode(random_bytes(32));"
define('ENCRYPTION_KEY', 'YOUR_GENERATED_KEY_HERE');

// Update site URLs
define('SITE_URL', 'https://yourdomain.com/payment_gateway');

// Add your Stripe keys
define('STRIPE_PUBLISHABLE_KEY', 'pk_test_YOUR_KEY');
define('STRIPE_SECRET_KEY', 'sk_test_YOUR_KEY');
define('STRIPE_WEBHOOK_SECRET', 'whsec_YOUR_WEBHOOK_SECRET');

// In production, set these to true/false appropriately
define('ENABLE_HTTPS_ONLY', true);
define('DISPLAY_ERRORS', false);
```

### Step 3: File Permissions

```bash
chmod 755 payment_gateway/
chmod 644 config.php
chmod 755 admin/
chmod 755 classes/
```

### Step 4: Create Log Directory

```bash
mkdir logs
chmod 755 logs
```

### Step 5: Default Admin Login

**Username:** Creatorschoice  
**Password:** Creators@7620  

⚠️ **IMPORTANT:** Change this password immediately after first login!

## 🔧 Configuration

### Setting Up Stripe

1. Create a Stripe account at https://stripe.com
2. Get your API keys from Dashboard → Developers → API keys
3. Add keys to `config.php`
4. For webhooks (optional):
   - Create webhook endpoint in Stripe Dashboard
   - Point to: `https://yourdomain.com/payment_gateway/webhook.php`
   - Select events: `payment_intent.succeeded`, `payment_intent.payment_failed`

### Gateway Configurations

1. Login to admin panel
2. Go to "Gateway Configurations"
3. Create new configuration with:
   - Configuration name
   - API keys (encrypted automatically)
   - Mode (test/live)
   - Business details
   - Default currency

### Creating Payment Links

1. Go to "Create Payment Link"
2. Fill in:
   - Payment title and description
   - Amount and currency
   - Gateway configuration
   - Optional: Max uses, expiration date, redirect URL
   - Optional: Recurring payment settings
3. Copy the generated link and share with customers

## 🔒 Security Best Practices

### Production Deployment

1. **Enable HTTPS:**
   ```php
   define('ENABLE_HTTPS_ONLY', true);
   ```

2. **Disable error display:**
   ```php
   define('DISPLAY_ERRORS', false);
   ```

3. **Move config.php outside web root:**
   - Move `config.php` to a directory above your web root
   - Update the require path in all files

4. **Strong passwords:**
   - All admin accounts must use strong passwords
   - Enforce password policy in user management

5. **Regular backups:**
   - Backup database regularly
   - Backup encryption keys securely

6. **Update regularly:**
   - Keep PHP and MySQL updated
   - Monitor security advisories

### Encryption Key Management

The encryption key is critical. If lost:
- You cannot decrypt stored API keys
- You'll need to re-enter all gateway configurations

**Backup your encryption key securely!**

## 📱 API Integration (Optional)

You can integrate this gateway into your applications:

```javascript
// Example: Creating a payment via API
fetch('https://yourdomain.com/payment_gateway/process-payment.php', {
    method: 'POST',
    headers: {
        'Content-Type': 'application/json'
    },
    body: JSON.stringify({
        payment_method_id: 'pm_xxxxx',
        link_id: 'PAY-XXXXX',
        customer_name: 'John Doe',
        customer_email: 'john@example.com'
    })
})
.then(response => response.json())
.then(data => {
    if (data.success) {
        console.log('Payment successful:', data.transaction_id);
    }
});
```

## 🎨 Customization

### Styling

Edit the following files:
- `admin/css/admin-style.css` - Admin panel styles
- `pay.php` - Customer payment page (inline styles)

### Adding Payment Processors

1. Create new class in `classes/` (e.g., `PayPalGateway.php`)
2. Implement the same interface as `PaymentGateway.php`
3. Update configuration options in admin panel

## 📊 Database Schema

### Tables:
- `admin_users` - Admin user accounts
- `gateway_config` - Payment gateway configurations
- `payment_links` - Generated payment links
- `payment_transactions` - Transaction records (minimal data)
- `admin_sessions` - Session management

## 🐛 Troubleshooting

### "Database connection failed"
- Check database credentials in `config.php`
- Ensure MySQL is running
- Verify user has proper permissions

### "Payment processing failed"
- Check Stripe API keys are correct
- Verify Stripe account is active
- Check error logs in `logs/error.log`

### "Session expired"
- Clear browser cookies
- Check session settings in `php.ini`
- Verify file permissions on session directory

### Payment page not loading
- Check `.htaccess` if using Apache
- Verify mod_rewrite is enabled
- Check web server error logs

## 📝 File Structure

```
payment_gateway/
├── admin/
│   ├── css/
│   │   └── admin-style.css
│   ├── includes/
│   │   ├── header.php
│   │   └── sidebar.php
│   ├── configurations.php
│   ├── create-link.php
│   ├── dashboard.php
│   ├── login.php
│   ├── logout.php
│   └── transactions.php
├── classes/
│   ├── AdminAuth.php
│   ├── Database.php
│   ├── PaymentGateway.php
│   └── Security.php
├── logs/
│   └── error.log
├── config.php
├── database.sql
├── pay.php
├── payment-success.php
├── process-payment.php
└── README.md
```

## 🔐 Compliance

This system is designed to be:
- **PCI DSS Compliant** - No card data stored
- **GDPR Ready** - Minimal personal data collection
- **SOC 2 Compatible** - Audit logging available

## 📄 License

This payment gateway system is provided as-is for educational and commercial use.

## ⚠️ Disclaimer

- Always test thoroughly before using in production
- Consult with legal and security professionals
- Comply with your local payment processing regulations
- Keep all software updated
- Monitor for security vulnerabilities

## 🆘 Support

For issues or questions:
1. Check the troubleshooting section
2. Review error logs in `logs/error.log`
3. Verify configuration settings
4. Test with Stripe test mode first

## 🔄 Updates

### Version 1.0.0
- Initial release
- Stripe integration
- Admin panel
- Payment link generation
- Transaction management
- Multi-currency support

---

**Made with 🔒 Security in mind**
