<?php
require_once '../config.php';
require_once '../classes/Database.php';
require_once '../classes/Security.php';
require_once '../classes/AdminAuth.php';

$auth = new AdminAuth();
$auth->requirePermission('admin');

$currentUser = $auth->getCurrentUser();
$db = Database::getInstance();

$success = '';
$error = '';

// Handle configuration creation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'create') {
    if (Security::verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        try {
            $configData = [
                'config_name' => Security::sanitize($_POST['config_name']),
                'api_key' => Security::encrypt($_POST['api_key']),
                'api_secret' => Security::encrypt($_POST['api_secret']),
                'webhook_secret' => !empty($_POST['webhook_secret']) ? Security::encrypt($_POST['webhook_secret']) : null,
                'currency' => strtoupper($_POST['currency']),
                'mode' => $_POST['mode'],
                'business_name' => Security::sanitize($_POST['business_name']),
                'business_email' => Security::sanitize($_POST['business_email']),
                'success_url' => $_POST['success_url'] ?? null,
                'cancel_url' => $_POST['cancel_url'] ?? null,
                'created_by' => $_SESSION['admin_id']
            ];
            
            $db->insert('gateway_config', $configData);
            $success = 'Configuration created successfully!';
            
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    } else {
        $error = 'Invalid security token';
    }
}

// Handle configuration toggle
if (isset($_GET['toggle']) && isset($_GET['id'])) {
    if ($auth->hasPermission('admin')) {
        $configId = intval($_GET['id']);
        $currentStatus = $db->fetchOne(
            "SELECT is_active FROM gateway_config WHERE id = ?",
            [$configId]
        )['is_active'];
        
        $db->update(
            'gateway_config',
            ['is_active' => $currentStatus ? 0 : 1],
            'id = ?',
            [$configId]
        );
        
        header('Location: configurations.php');
        exit;
    }
}

// Get all configurations
$configurations = $db->fetchAll("
    SELECT gc.*, au.username as created_by_name
    FROM gateway_config gc
    LEFT JOIN admin_users au ON gc.created_by = au.id
    ORDER BY gc.created_at DESC
");

$csrfToken = Security::generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gateway Configurations - Payment Gateway Admin</title>
    <link rel="stylesheet" href="css/admin-style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>Gateway Configurations</h1>
                <p>Manage payment gateway API settings</p>
            </div>
            
            <?php if ($success): ?>
            <div class="alert alert-success">
                <strong>Success!</strong> <?php echo Security::escape($success); ?>
            </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
            <div class="alert alert-error">
                <strong>Error!</strong> <?php echo Security::escape($error); ?>
            </div>
            <?php endif; ?>
            
            <div class="section-card" style="margin-bottom: 30px;">
                <h2>Create New Configuration</h2>
                <form method="POST" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">
                    <input type="hidden" name="action" value="create">
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="form-group">
                            <label for="config_name">Configuration Name *</label>
                            <input type="text" id="config_name" name="config_name" class="form-control" required 
                                   placeholder="e.g., Production Config">
                        </div>
                        
                        <div class="form-group">
                            <label for="mode">Mode *</label>
                            <select id="mode" name="mode" class="form-select" required>
                                <option value="test">Test Mode</option>
                                <option value="live">Live Mode</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="api_key">API Publishable Key *</label>
                        <input type="text" id="api_key" name="api_key" class="form-control" required 
                               placeholder="pk_test_...">
                    </div>
                    
                    <div class="form-group">
                        <label for="api_secret">API Secret Key *</label>
                        <input type="password" id="api_secret" name="api_secret" class="form-control" required 
                               placeholder="sk_test_...">
                    </div>
                    
                    <div class="form-group">
                        <label for="webhook_secret">Webhook Secret (Optional)</label>
                        <input type="password" id="webhook_secret" name="webhook_secret" class="form-control" 
                               placeholder="whsec_...">
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="form-group">
                            <label for="business_name">Business Name *</label>
                            <input type="text" id="business_name" name="business_name" class="form-control" required 
                                   placeholder="Your Business Name">
                        </div>
                        
                        <div class="form-group">
                            <label for="business_email">Business Email *</label>
                            <input type="email" id="business_email" name="business_email" class="form-control" required 
                                   placeholder="business@example.com">
                        </div>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="form-group">
                            <label for="currency">Default Currency *</label>
                            <select id="currency" name="currency" class="form-select" required>
                                <option value="USD">USD - US Dollar</option>
                                <option value="EUR">EUR - Euro</option>
                                <option value="GBP">GBP - British Pound</option>
                                <option value="CAD">CAD - Canadian Dollar</option>
                                <option value="AUD">AUD - Australian Dollar</option>
                                <option value="INR">INR - Indian Rupee</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="success_url">Success Redirect URL</label>
                            <input type="url" id="success_url" name="success_url" class="form-control" 
                                   placeholder="https://yoursite.com/success">
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        Create Configuration
                    </button>
                </form>
            </div>
            
            <div class="section-card">
                <h2>Existing Configurations</h2>
                
                <?php if (empty($configurations)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">
                    No configurations found. Create your first configuration above.
                </p>
                <?php else: ?>
                <div class="data-table">
                    <table>
                        <thead>
                            <tr>
                                <th>Configuration Name</th>
                                <th>Mode</th>
                                <th>Currency</th>
                                <th>Business Name</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($configurations as $config): ?>
                            <tr>
                                <td>
                                    <strong><?php echo Security::escape($config['config_name']); ?></strong>
                                </td>
                                <td>
                                    <span class="badge badge-<?php echo $config['mode'] === 'live' ? 'success' : 'warning'; ?>">
                                        <?php echo ucfirst($config['mode']); ?>
                                    </span>
                                </td>
                                <td><?php echo Security::escape($config['currency']); ?></td>
                                <td><?php echo Security::escape($config['business_name']); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $config['is_active'] ? 'success' : 'danger'; ?>">
                                        <?php echo $config['is_active'] ? 'Active' : 'Inactive'; ?>
                                    </span>
                                </td>
                                <td><?php echo date('M d, Y', strtotime($config['created_at'])); ?></td>
                                <td>
                                    <a href="?toggle=1&id=<?php echo $config['id']; ?>" 
                                       class="btn btn-secondary" style="padding: 6px 12px; font-size: 12px;">
                                        <?php echo $config['is_active'] ? 'Deactivate' : 'Activate'; ?>
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</body>
</html>
