<?php
require_once '../config.php';
require_once '../classes/Database.php';
require_once '../classes/Security.php';
require_once '../classes/AdminAuth.php';
require_once '../classes/PaymentGateway.php';

$auth = new AdminAuth();
$auth->requirePermission('admin');

$currentUser = $auth->getCurrentUser();
$db = Database::getInstance();
$gateway = new PaymentGateway();

$success = '';
$error = '';
$createdLink = null;

// Get available configurations
$configs = $db->fetchAll("SELECT * FROM gateway_config WHERE is_active = 1");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (Security::verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        try {
            $linkData = [
                'title' => $_POST['title'],
                'description' => $_POST['description'],
                'amount' => $_POST['amount'],
                'currency' => $_POST['currency'],
                'config_id' => $_POST['config_id'],
                'is_recurring' => isset($_POST['is_recurring']) ? 1 : 0,
                'recurring_interval' => $_POST['recurring_interval'] ?? null,
                'max_uses' => !empty($_POST['max_uses']) ? $_POST['max_uses'] : null,
                'expires_at' => !empty($_POST['expires_at']) ? $_POST['expires_at'] : null,
                'redirect_url' => $_POST['redirect_url'] ?? null,
                'custom_fields' => []
            ];
            
            $createdLink = $gateway->createPaymentLink($linkData);
            $success = 'Payment link created successfully!';
            
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    } else {
        $error = 'Invalid security token';
    }
}

$csrfToken = Security::generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Payment Link - Payment Gateway Admin</title>
    <link rel="stylesheet" href="css/admin-style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>Create Payment Link</h1>
                <p>Generate a new payment link for customers</p>
            </div>
            
            <?php if ($success): ?>
            <div class="alert alert-success">
                <strong>Success!</strong> <?php echo Security::escape($success); ?>
            </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
            <div class="alert alert-error">
                <strong>Error!</strong> <?php echo Security::escape($error); ?>
            </div>
            <?php endif; ?>
            
            <?php if ($createdLink): ?>
            <div class="section-card" style="margin-bottom: 30px; background: #f0fdf4; border: 2px solid #10b981;">
                <h2 style="color: #059669; margin-bottom: 15px;">🎉 Payment Link Created!</h2>
                <div style="background: white; padding: 20px; border-radius: 8px; margin-bottom: 15px;">
                    <p style="margin-bottom: 10px; font-weight: 600;">Payment Link:</p>
                    <div style="display: flex; gap: 10px;">
                        <input type="text" id="paymentLinkUrl" value="<?php echo Security::escape($createdLink['url']); ?>" readonly 
                               style="flex: 1; padding: 12px; border: 2px solid #e5e7eb; border-radius: 6px; font-family: monospace;">
                        <button onclick="copyToClipboard()" class="btn btn-primary">Copy Link</button>
                    </div>
                </div>
                <p style="color: #065f46;">Link ID: <strong><?php echo Security::escape($createdLink['link_id']); ?></strong></p>
                <a href="<?php echo Security::escape($createdLink['url']); ?>" target="_blank" class="btn btn-secondary" style="margin-top: 15px;">
                    Test Payment Link
                </a>
            </div>
            <?php endif; ?>
            
            <div class="section-card">
                <form method="POST" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">
                    
                    <div class="form-group">
                        <label for="title">Payment Title *</label>
                        <input type="text" id="title" name="title" class="form-control" required 
                               placeholder="e.g., Product Purchase, Subscription Payment">
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea id="description" name="description" class="form-control" rows="3" 
                                  placeholder="Optional description of the payment"></textarea>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="form-group">
                            <label for="amount">Amount *</label>
                            <input type="number" id="amount" name="amount" class="form-control" step="0.01" min="0.01" required 
                                   placeholder="0.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="currency">Currency *</label>
                            <select id="currency" name="currency" class="form-select" required>
                                <option value="USD">USD - US Dollar</option>
                                <option value="EUR">EUR - Euro</option>
                                <option value="GBP">GBP - British Pound</option>
                                <option value="CAD">CAD - Canadian Dollar</option>
                                <option value="AUD">AUD - Australian Dollar</option>
                                <option value="JPY">JPY - Japanese Yen</option>
                                <option value="INR">INR - Indian Rupee</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="config_id">Gateway Configuration *</label>
                        <select id="config_id" name="config_id" class="form-select" required>
                            <option value="">Select Configuration</option>
                            <?php foreach ($configs as $config): ?>
                            <option value="<?php echo $config['id']; ?>">
                                <?php echo Security::escape($config['config_name']); ?> 
                                (<?php echo ucfirst($config['mode']); ?>)
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="form-group">
                            <label for="max_uses">Maximum Uses (Optional)</label>
                            <input type="number" id="max_uses" name="max_uses" class="form-control" min="1" 
                                   placeholder="Leave empty for unlimited">
                        </div>
                        
                        <div class="form-group">
                            <label for="expires_at">Expiration Date (Optional)</label>
                            <input type="datetime-local" id="expires_at" name="expires_at" class="form-control">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="redirect_url">Success Redirect URL (Optional)</label>
                        <input type="url" id="redirect_url" name="redirect_url" class="form-control" 
                               placeholder="https://yoursite.com/thank-you">
                    </div>
                    
                    <div class="form-group">
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="checkbox" id="is_recurring" name="is_recurring" value="1" 
                                   onchange="toggleRecurring()">
                            <span>Recurring Payment</span>
                        </label>
                    </div>
                    
                    <div id="recurringOptions" style="display: none;">
                        <div class="form-group">
                            <label for="recurring_interval">Recurring Interval *</label>
                            <select id="recurring_interval" name="recurring_interval" class="form-select">
                                <option value="daily">Daily</option>
                                <option value="weekly">Weekly</option>
                                <option value="monthly">Monthly</option>
                                <option value="yearly">Yearly</option>
                            </select>
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 12px; margin-top: 30px;">
                        <button type="submit" class="btn btn-primary">
                            Create Payment Link
                        </button>
                        <a href="payment-links.php" class="btn btn-secondary">
                            Cancel
                        </a>
                    </div>
                </form>
            </div>
        </main>
    </div>
    
    <script>
        function toggleRecurring() {
            const checkbox = document.getElementById('is_recurring');
            const options = document.getElementById('recurringOptions');
            options.style.display = checkbox.checked ? 'block' : 'none';
        }
        
        function copyToClipboard() {
            const input = document.getElementById('paymentLinkUrl');
            input.select();
            document.execCommand('copy');
            alert('Payment link copied to clipboard!');
        }
    </script>
</body>
</html>
