<?php
require_once '../config.php';
require_once '../classes/Database.php';
require_once '../classes/Security.php';
require_once '../classes/AdminAuth.php';

$auth = new AdminAuth();
$auth->requireLogin();

$currentUser = $auth->getCurrentUser();
$db = Database::getInstance();

$success = '';
$error = '';

// Handle link deletion
if (isset($_GET['delete']) && isset($_GET['id']) && $auth->hasPermission('admin')) {
    $linkId = Security::sanitize($_GET['id']);
    $db->update('payment_links', ['is_active' => 0], 'link_id = ?', [$linkId]);
    $success = 'Payment link deactivated successfully!';
}

// Handle link toggle
if (isset($_GET['toggle']) && isset($_GET['id']) && $auth->hasPermission('admin')) {
    $linkId = Security::sanitize($_GET['id']);
    $current = $db->fetchOne("SELECT is_active FROM payment_links WHERE link_id = ?", [$linkId]);
    if ($current) {
        $db->update('payment_links', ['is_active' => $current['is_active'] ? 0 : 1], 'link_id = ?', [$linkId]);
        header('Location: payment-links.php');
        exit;
    }
}

// Get all payment links
$links = $db->fetchAll("
    SELECT 
        pl.*,
        gc.config_name,
        gc.mode,
        au.username as created_by_name
    FROM payment_links pl
    LEFT JOIN gateway_config gc ON pl.config_id = gc.id
    LEFT JOIN admin_users au ON pl.created_by = au.id
    ORDER BY pl.created_at DESC
");

$highlight = $_GET['highlight'] ?? null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Links - Payment Gateway Admin</title>
    <link rel="stylesheet" href="css/admin-style.css">
    <style>
        .link-card {
            background: white;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            transition: all 0.2s;
        }
        
        .link-card:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            border-color: #667eea;
        }
        
        .link-card.highlight {
            border-color: #10b981;
            background: #f0fdf4;
        }
        
        .link-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }
        
        .link-title {
            font-size: 20px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 5px;
        }
        
        .link-id {
            font-family: monospace;
            font-size: 12px;
            color: #6b7280;
            background: #f3f4f6;
            padding: 4px 8px;
            border-radius: 4px;
        }
        
        .link-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .stat-item {
            display: flex;
            flex-direction: column;
        }
        
        .stat-label {
            font-size: 12px;
            color: #6b7280;
            margin-bottom: 4px;
        }
        
        .stat-value {
            font-size: 16px;
            font-weight: 600;
            color: #1f2937;
        }
        
        .link-url-box {
            background: #f9fafb;
            padding: 12px;
            border-radius: 8px;
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .link-url {
            flex: 1;
            font-family: monospace;
            font-size: 13px;
            color: #667eea;
            word-break: break-all;
        }
        
        .btn-copy {
            padding: 8px 16px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            white-space: nowrap;
        }
        
        .btn-copy:hover {
            background: #5568d3;
        }
        
        .link-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .btn-action {
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 500;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-view {
            background: #10b981;
            color: white;
        }
        
        .btn-toggle {
            background: #f59e0b;
            color: white;
        }
        
        .btn-delete {
            background: #ef4444;
            color: white;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>Payment Links</h1>
                <p>Manage all your payment links</p>
            </div>
            
            <?php if ($success): ?>
            <div class="alert alert-success">
                <strong>Success!</strong> <?php echo Security::escape($success); ?>
            </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
            <div class="alert alert-error">
                <strong>Error!</strong> <?php echo Security::escape($error); ?>
            </div>
            <?php endif; ?>
            
            <div style="margin-bottom: 20px;">
                <a href="create-link.php" class="btn btn-primary">
                    + Create New Payment Link
                </a>
            </div>
            
            <?php if (empty($links)): ?>
            <div class="section-card">
                <p style="text-align: center; color: #666; padding: 40px;">
                    No payment links found. Create your first payment link to get started!
                </p>
            </div>
            <?php else: ?>
            
            <?php foreach ($links as $link): ?>
            <div class="link-card <?php echo $highlight === $link['link_id'] ? 'highlight' : ''; ?>">
                <div class="link-header">
                    <div>
                        <div class="link-title"><?php echo Security::escape($link['title']); ?></div>
                        <span class="link-id"><?php echo Security::escape($link['link_id']); ?></span>
                        <?php if (!$link['is_active']): ?>
                        <span class="badge badge-danger" style="margin-left: 10px;">Inactive</span>
                        <?php endif; ?>
                    </div>
                    <div>
                        <span class="badge badge-<?php echo $link['mode'] === 'live' ? 'success' : 'warning'; ?>">
                            <?php echo ucfirst($link['mode']); ?>
                        </span>
                    </div>
                </div>
                
                <?php if ($link['description']): ?>
                <p style="color: #6b7280; margin-bottom: 15px;">
                    <?php echo Security::escape($link['description']); ?>
                </p>
                <?php endif; ?>
                
                <div class="link-stats">
                    <div class="stat-item">
                        <span class="stat-label">Amount</span>
                        <span class="stat-value">
                            <?php echo $link['currency']; ?> <?php echo number_format($link['amount'], 2); ?>
                        </span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Uses</span>
                        <span class="stat-value">
                            <?php echo $link['current_uses']; ?> 
                            <?php if ($link['max_uses']): ?>
                            / <?php echo $link['max_uses']; ?>
                            <?php endif; ?>
                        </span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Expires</span>
                        <span class="stat-value">
                            <?php echo $link['expires_at'] ? date('M d, Y', strtotime($link['expires_at'])) : 'Never'; ?>
                        </span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Created</span>
                        <span class="stat-value">
                            <?php echo date('M d, Y', strtotime($link['created_at'])); ?>
                        </span>
                    </div>
                </div>
                
                <div class="link-url-box">
                    <div class="link-url" id="url-<?php echo $link['id']; ?>">
                        <?php echo SITE_URL; ?>/pay.php?id=<?php echo Security::escape($link['link_id']); ?>
                    </div>
                    <button class="btn-copy" onclick="copyLink(<?php echo $link['id']; ?>)">
                        Copy Link
                    </button>
                </div>
                
                <div class="link-actions">
                    <a href="<?php echo SITE_URL; ?>/pay.php?id=<?php echo Security::escape($link['link_id']); ?>" 
                       target="_blank" class="btn-action btn-view">
                        View Payment Page
                    </a>
                    
                    <?php if ($auth->hasPermission('admin')): ?>
                    <a href="?toggle=1&id=<?php echo Security::escape($link['link_id']); ?>" 
                       class="btn-action btn-toggle">
                        <?php echo $link['is_active'] ? 'Deactivate' : 'Activate'; ?>
                    </a>
                    
                    <a href="?delete=1&id=<?php echo Security::escape($link['link_id']); ?>" 
                       class="btn-action btn-delete"
                       onclick="return confirm('Are you sure you want to deactivate this payment link?')">
                        Deactivate
                    </a>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
            
            <?php endif; ?>
        </main>
    </div>
    
    <script>
        function copyLink(id) {
            const urlElement = document.getElementById('url-' + id);
            const text = urlElement.textContent.trim();
            
            // Create temporary textarea
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            
            // Select and copy
            textarea.select();
            document.execCommand('copy');
            
            // Remove textarea
            document.body.removeChild(textarea);
            
            // Show feedback
            alert('Payment link copied to clipboard!');
        }
    </script>
</body>
</html>
