<?php
require_once '../config.php';
require_once '../classes/Database.php';
require_once '../classes/Security.php';
require_once '../classes/AdminAuth.php';
require_once '../classes/PaymentGateway.php';

$auth = new AdminAuth();
$auth->requireLogin();

$currentUser = $auth->getCurrentUser();
$gateway = new PaymentGateway();

// Get filter parameters
$status = $_GET['status'] ?? null;
$limit = 50;
$offset = isset($_GET['page']) ? (intval($_GET['page']) - 1) * $limit : 0;
$currentPage = isset($_GET['page']) ? intval($_GET['page']) : 1;

// Get transactions
$transactions = $gateway->getTransactions($limit, $offset, $status);

// Get total count for pagination
$db = Database::getInstance();
$whereClause = $status ? "WHERE status = ?" : "WHERE 1=1";
$params = $status ? [$status] : [];
$totalCount = $db->fetchOne(
    "SELECT COUNT(*) as count FROM payment_transactions {$whereClause}",
    $params
)['count'];

$totalPages = ceil($totalCount / $limit);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transactions - Payment Gateway Admin</title>
    <link rel="stylesheet" href="css/admin-style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>Payment Transactions</h1>
                <p>View and manage payment history</p>
            </div>
            
            <div class="section-card" style="margin-bottom: 20px;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>Filter by Status:</strong>
                    </div>
                    <div style="display: flex; gap: 10px;">
                        <a href="transactions.php" class="btn <?php echo !$status ? 'btn-primary' : 'btn-secondary'; ?>" 
                           style="padding: 8px 16px; font-size: 14px;">
                            All
                        </a>
                        <a href="transactions.php?status=completed" 
                           class="btn <?php echo $status === 'completed' ? 'btn-primary' : 'btn-secondary'; ?>" 
                           style="padding: 8px 16px; font-size: 14px;">
                            Completed
                        </a>
                        <a href="transactions.php?status=pending" 
                           class="btn <?php echo $status === 'pending' ? 'btn-primary' : 'btn-secondary'; ?>" 
                           style="padding: 8px 16px; font-size: 14px;">
                            Pending
                        </a>
                        <a href="transactions.php?status=failed" 
                           class="btn <?php echo $status === 'failed' ? 'btn-primary' : 'btn-secondary'; ?>" 
                           style="padding: 8px 16px; font-size: 14px;">
                            Failed
                        </a>
                    </div>
                </div>
            </div>
            
            <div class="section-card">
                <h2>Transaction History</h2>
                
                <?php if (empty($transactions)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">
                    No transactions found.
                </p>
                <?php else: ?>
                <div class="data-table">
                    <table>
                        <thead>
                            <tr>
                                <th>Transaction ID</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Card</th>
                                <th>Country</th>
                                <th>Link ID</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($transactions as $txn): ?>
                            <tr>
                                <td>
                                    <strong style="font-family: monospace;">
                                        <?php echo Security::escape($txn['transaction_id']); ?>
                                    </strong>
                                </td>
                                <td>
                                    <strong>
                                        <?php echo $txn['currency']; ?> 
                                        <?php echo number_format($txn['amount'], 2); ?>
                                    </strong>
                                </td>
                                <td>
                                    <?php
                                    $badgeClass = 'info';
                                    switch ($txn['status']) {
                                        case 'completed':
                                            $badgeClass = 'success';
                                            break;
                                        case 'failed':
                                        case 'cancelled':
                                            $badgeClass = 'danger';
                                            break;
                                        case 'pending':
                                        case 'processing':
                                            $badgeClass = 'warning';
                                            break;
                                    }
                                    ?>
                                    <span class="badge badge-<?php echo $badgeClass; ?>">
                                        <?php echo ucfirst($txn['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($txn['card_brand'] && $txn['card_last4']): ?>
                                        <?php echo ucfirst($txn['card_brand']); ?> 
                                        •••• <?php echo $txn['card_last4']; ?>
                                    <?php else: ?>
                                        <span style="color: #999;">N/A</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php echo $txn['country_code'] ? Security::escape($txn['country_code']) : '-'; ?>
                                </td>
                                <td>
                                    <?php if ($txn['link_id']): ?>
                                        <a href="payment-links.php?highlight=<?php echo $txn['link_id']; ?>" 
                                           style="color: #667eea; text-decoration: none;">
                                            <?php echo Security::escape($txn['link_id']); ?>
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div><?php echo date('M d, Y', strtotime($txn['created_at'])); ?></div>
                                    <div style="font-size: 11px; color: #999;">
                                        <?php echo date('H:i:s', strtotime($txn['created_at'])); ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if ($totalPages > 1): ?>
                <div style="display: flex; justify-content: center; gap: 10px; margin-top: 20px; padding-top: 20px; border-top: 1px solid #e5e7eb;">
                    <?php if ($currentPage > 1): ?>
                    <a href="?page=<?php echo $currentPage - 1; ?><?php echo $status ? '&status=' . $status : ''; ?>" 
                       class="btn btn-secondary">
                        Previous
                    </a>
                    <?php endif; ?>
                    
                    <span style="padding: 8px 16px; color: #666;">
                        Page <?php echo $currentPage; ?> of <?php echo $totalPages; ?>
                    </span>
                    
                    <?php if ($currentPage < $totalPages): ?>
                    <a href="?page=<?php echo $currentPage + 1; ?><?php echo $status ? '&status=' . $status : ''; ?>" 
                       class="btn btn-secondary">
                        Next
                    </a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </div>
            
            <div class="alert alert-info" style="margin-top: 20px;">
                <strong>🔒 Privacy Notice:</strong> Transaction details are encrypted and only minimal information is stored. 
                Full card details are never stored in our database. We comply with PCI DSS standards.
            </div>
        </main>
    </div>
</body>
</html>
