<?php
class AdminAuth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->startSecureSession();
    }
    
    private function startSecureSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }
    
    // Admin login
    public function login($username, $password) {
        $ip = Security::getClientIP();
        
        // Check rate limiting
        if (!Security::checkRateLimit('login_' . $ip, MAX_LOGIN_ATTEMPTS, LOCKOUT_TIME)) {
            return [
                'success' => false,
                'message' => 'Too many login attempts. Please try again later.'
            ];
        }
        
        // Sanitize input
        $username = Security::sanitize($username);
        
        // Get user from database
        $user = $this->db->fetchOne(
            "SELECT * FROM admin_users WHERE username = ? AND is_active = 1",
            [$username]
        );
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'Invalid username or password'
            ];
        }
        
        // Verify password
        if (!Security::verifyPassword($password, $user['password'])) {
            return [
                'success' => false,
                'message' => 'Invalid username or password'
            ];
        }
        
        // Create session
        $this->createSession($user);
        
        // Update last login
        $this->db->update(
            'admin_users',
            ['last_login' => date('Y-m-d H:i:s')],
            'id = ?',
            [$user['id']]
        );
        
        return [
            'success' => true,
            'message' => 'Login successful',
            'user' => [
                'id' => $user['id'],
                'username' => $user['username'],
                'role' => $user['role']
            ]
        ];
    }
    
    // Create admin session
    private function createSession($user) {
        session_regenerate_id(true);
        
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_id'] = $user['id'];
        $_SESSION['admin_username'] = $user['username'];
        $_SESSION['admin_role'] = $user['role'];
        $_SESSION['session_start'] = time();
        
        // Create session token in database
        $sessionToken = Security::generateToken();
        $expiresAt = date('Y-m-d H:i:s', time() + SESSION_LIFETIME);
        
        $this->db->insert('admin_sessions', [
            'user_id' => $user['id'],
            'session_token' => $sessionToken,
            'ip_address' => Security::getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'expires_at' => $expiresAt
        ]);
        
        $_SESSION['session_token'] = $sessionToken;
    }
    
    // Check if admin is logged in
    public function isLoggedIn() {
        if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
            return false;
        }
        
        // Check session timeout
        if (isset($_SESSION['session_start'])) {
            if (time() - $_SESSION['session_start'] > SESSION_LIFETIME) {
                $this->logout();
                return false;
            }
        }
        
        // Verify session token
        if (isset($_SESSION['session_token'])) {
            $session = $this->db->fetchOne(
                "SELECT * FROM admin_sessions WHERE session_token = ? AND expires_at > NOW()",
                [$_SESSION['session_token']]
            );
            
            if (!$session) {
                $this->logout();
                return false;
            }
        }
        
        return true;
    }
    
    // Get current admin user
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return $this->db->fetchOne(
            "SELECT id, username, email, full_name, role FROM admin_users WHERE id = ?",
            [$_SESSION['admin_id']]
        );
    }
    
    // Logout
    public function logout() {
        if (isset($_SESSION['session_token'])) {
            $this->db->delete(
                'admin_sessions',
                'session_token = ?',
                [$_SESSION['session_token']]
            );
        }
        
        session_unset();
        session_destroy();
    }
    
    // Check permission
    public function hasPermission($requiredRole) {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        $roleHierarchy = [
            'viewer' => 1,
            'admin' => 2,
            'super_admin' => 3
        ];
        
        $userRole = $_SESSION['admin_role'];
        
        return $roleHierarchy[$userRole] >= $roleHierarchy[$requiredRole];
    }
    
    // Require login
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: /payment_gateway/admin/login.php');
            exit;
        }
    }
    
    // Require permission
    public function requirePermission($role) {
        $this->requireLogin();
        
        if (!$this->hasPermission($role)) {
            die('Access denied. Insufficient permissions.');
        }
    }
    
    // Clean expired sessions
    public function cleanExpiredSessions() {
        $this->db->query("DELETE FROM admin_sessions WHERE expires_at < NOW()");
    }
}
