<?php
class PaymentGateway {
    private $db;
    private $config;
    
    public function __construct($configId = null) {
        $this->db = Database::getInstance();
        
        if ($configId) {
            $this->loadConfig($configId);
        }
    }
    
    // Load gateway configuration
    private function loadConfig($configId) {
        $this->config = $this->db->fetchOne(
            "SELECT * FROM gateway_config WHERE id = ? AND is_active = 1",
            [$configId]
        );
        
        if (!$this->config) {
            throw new Exception("Gateway configuration not found");
        }
    }
    
    // Create payment intent (Stripe)
    public function createPaymentIntent($amount, $currency, $metadata = []) {
        if (!$this->config) {
            throw new Exception("Gateway not configured");
        }
        
        $apiKey = Security::decrypt($this->config['api_secret']);
        
        $data = [
            'amount' => $amount * 100, // Convert to cents
            'currency' => strtolower($currency),
            'payment_method_types' => ['card'],
            'metadata' => $metadata
        ];
        
        $response = $this->stripeAPICall('payment_intents', $data, $apiKey);
        
        return $response;
    }
    
    // Stripe API call
    private function stripeAPICall($endpoint, $data, $apiKey) {
        $url = "https://api.stripe.com/v1/{$endpoint}";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/x-www-form-urlencoded'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new Exception("API call failed: " . $error);
        }
        
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($httpCode >= 400) {
            throw new Exception($result['error']['message'] ?? 'Payment processing failed');
        }
        
        return $result;
    }
    
    // Create payment link
    public function createPaymentLink($data) {
        $linkId = 'PAY-' . strtoupper(Security::generateSecureCode(12));
        
        $insertData = [
            'link_id' => $linkId,
            'title' => Security::sanitize($data['title']),
            'description' => Security::sanitize($data['description'] ?? ''),
            'amount' => floatval($data['amount']),
            'currency' => strtoupper($data['currency'] ?? 'USD'),
            'config_id' => intval($data['config_id']),
            'is_recurring' => intval($data['is_recurring'] ?? 0),
            'recurring_interval' => $data['recurring_interval'] ?? null,
            'max_uses' => $data['max_uses'] ?? null,
            'expires_at' => $data['expires_at'] ?? null,
            'custom_fields' => json_encode($data['custom_fields'] ?? []),
            'redirect_url' => $data['redirect_url'] ?? null,
            'created_by' => $_SESSION['admin_id'] ?? null
        ];
        
        $id = $this->db->insert('payment_links', $insertData);
        
        return [
            'id' => $id,
            'link_id' => $linkId,
            'url' => SITE_URL . '/pay.php?id=' . $linkId
        ];
    }
    
    // Get payment link
    public function getPaymentLink($linkId) {
        $link = $this->db->fetchOne(
            "SELECT pl.*, gc.business_name, gc.currency as default_currency, gc.mode
             FROM payment_links pl
             JOIN gateway_config gc ON pl.config_id = gc.id
             WHERE pl.link_id = ? AND pl.is_active = 1",
            [$linkId]
        );
        
        if (!$link) {
            return null;
        }
        
        // Check expiration
        if ($link['expires_at'] && strtotime($link['expires_at']) < time()) {
            return null;
        }
        
        // Check max uses
        if ($link['max_uses'] && $link['current_uses'] >= $link['max_uses']) {
            return null;
        }
        
        return $link;
    }
    
    // Process payment
    public function processPayment($linkId, $paymentMethodId, $customerData) {
        $link = $this->getPaymentLink($linkId);
        
        if (!$link) {
            throw new Exception("Payment link not found or expired");
        }
        
        // Load config
        $this->loadConfig($link['config_id']);
        
        // Create transaction record
        $transactionId = 'TXN-' . strtoupper(Security::generateSecureCode(16));
        
        $transactionData = [
            'transaction_id' => $transactionId,
            'link_id' => $linkId,
            'amount' => $link['amount'],
            'currency' => $link['currency'],
            'status' => 'pending',
            'customer_email_hash' => Security::hash($customerData['email']),
            'customer_ip_hash' => Security::hash(Security::getClientIP()),
            'metadata' => Security::encrypt(json_encode($customerData))
        ];
        
        $txnId = $this->db->insert('payment_transactions', $transactionData);
        
        try {
            // Create payment intent
            $intent = $this->createPaymentIntent(
                $link['amount'],
                $link['currency'],
                [
                    'transaction_id' => $transactionId,
                    'link_id' => $linkId
                ]
            );
            
            // Confirm payment
            $apiKey = Security::decrypt($this->config['api_secret']);
            $confirmation = $this->stripeAPICall(
                'payment_intents/' . $intent['id'] . '/confirm',
                ['payment_method' => $paymentMethodId],
                $apiKey
            );
            
            // Update transaction
            $status = $confirmation['status'] === 'succeeded' ? 'completed' : 'processing';
            
            $updateData = [
                'payment_intent_id' => $intent['id'],
                'status' => $status,
                'card_last4' => $confirmation['charges']['data'][0]['payment_method_details']['card']['last4'] ?? null,
                'card_brand' => $confirmation['charges']['data'][0]['payment_method_details']['card']['brand'] ?? null,
                'country_code' => $confirmation['charges']['data'][0]['payment_method_details']['card']['country'] ?? null,
                'processed_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->update('payment_transactions', $updateData, 'id = ?', [$txnId]);
            
            // Increment link uses
            $this->db->query(
                "UPDATE payment_links SET current_uses = current_uses + 1 WHERE link_id = ?",
                [$linkId]
            );
            
            return [
                'success' => true,
                'transaction_id' => $transactionId,
                'status' => $status,
                'redirect_url' => $link['redirect_url']
            ];
            
        } catch (Exception $e) {
            // Update transaction with error
            $this->db->update(
                'payment_transactions',
                [
                    'status' => 'failed',
                    'error_message' => $e->getMessage()
                ],
                'id = ?',
                [$txnId]
            );
            
            throw $e;
        }
    }
    
    // Get transaction stats (admin only)
    public function getTransactionStats($dateFrom = null, $dateTo = null) {
        $where = "1=1";
        $params = [];
        
        if ($dateFrom) {
            $where .= " AND created_at >= ?";
            $params[] = $dateFrom;
        }
        
        if ($dateTo) {
            $where .= " AND created_at <= ?";
            $params[] = $dateTo;
        }
        
        $stats = $this->db->fetchOne("
            SELECT 
                COUNT(*) as total_transactions,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as successful,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_amount,
                COUNT(DISTINCT DATE(created_at)) as days_active
            FROM payment_transactions
            WHERE {$where}
        ", $params);
        
        return $stats;
    }
    
    // Get transactions (admin only, no sensitive data)
    public function getTransactions($limit = 50, $offset = 0, $status = null) {
        $where = "1=1";
        $params = [];
        
        if ($status) {
            $where .= " AND status = ?";
            $params[] = $status;
        }
        
        $params[] = $limit;
        $params[] = $offset;
        
        return $this->db->fetchAll("
            SELECT 
                id,
                transaction_id,
                link_id,
                amount,
                currency,
                status,
                card_last4,
                card_brand,
                country_code,
                processed_at,
                created_at
            FROM payment_transactions
            WHERE {$where}
            ORDER BY created_at DESC
            LIMIT ? OFFSET ?
        ", $params);
    }
}
