<?php
/**
 * Installation Verification Script
 * Run this file in your browser to verify all requirements are met
 * Delete this file after successful installation
 */

$checks = [];
$errors = [];
$warnings = [];

// Check PHP Version
$checks['PHP Version'] = version_compare(PHP_VERSION, '7.4.0', '>=');
if (!$checks['PHP Version']) {
    $errors[] = 'PHP 7.4 or higher is required. Current version: ' . PHP_VERSION;
}

// Check required extensions
$required_extensions = ['pdo', 'pdo_mysql', 'curl', 'openssl', 'json', 'mbstring'];
foreach ($required_extensions as $ext) {
    $checks[$ext . ' extension'] = extension_loaded($ext);
    if (!$checks[$ext . ' extension']) {
        $errors[] = "Required PHP extension '{$ext}' is not loaded";
    }
}

// Check config file
$checks['config.php exists'] = file_exists(__DIR__ . '/config.php');
if (!$checks['config.php exists']) {
    $errors[] = 'config.php file not found';
}

// Check classes directory
$checks['classes/ directory'] = is_dir(__DIR__ . '/classes');
if (!$checks['classes/ directory']) {
    $errors[] = 'classes/ directory not found';
}

// Check admin directory
$checks['admin/ directory'] = is_dir(__DIR__ . '/admin');
if (!$checks['admin/ directory']) {
    $errors[] = 'admin/ directory not found';
}

// Check logs directory
$checks['logs/ directory'] = is_dir(__DIR__ . '/logs');
if (!$checks['logs/ directory']) {
    $warnings[] = 'logs/ directory not found - create it for error logging';
}

// Check logs directory writable
if (is_dir(__DIR__ . '/logs')) {
    $checks['logs/ writable'] = is_writable(__DIR__ . '/logs');
    if (!$checks['logs/ writable']) {
        $warnings[] = 'logs/ directory is not writable - chmod 755 logs/';
    }
}

// Try to include config
if (file_exists(__DIR__ . '/config.php')) {
    require_once __DIR__ . '/config.php';
    
    // Check database connection
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $pdo = new PDO($dsn, DB_USER, DB_PASS);
        $checks['Database Connection'] = true;
        
        // Check if tables exist
        $tables = ['admin_users', 'gateway_config', 'payment_links', 'payment_transactions'];
        foreach ($tables as $table) {
            $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
            $exists = $stmt->rowCount() > 0;
            $checks["Table: {$table}"] = $exists;
            if (!$exists) {
                $errors[] = "Database table '{$table}' not found. Import database.sql";
            }
        }
    } catch (PDOException $e) {
        $checks['Database Connection'] = false;
        $errors[] = 'Database connection failed: ' . $e->getMessage();
    }
    
    // Check encryption key
    $checks['Encryption Key Set'] = defined('ENCRYPTION_KEY') && strlen(ENCRYPTION_KEY) >= 32;
    if (!$checks['Encryption Key Set']) {
        $warnings[] = 'ENCRYPTION_KEY should be at least 32 characters. Generate with: php -r "echo base64_encode(random_bytes(32));"';
    }
    
    // Check Stripe keys
    $checks['Stripe Keys Set'] = defined('STRIPE_PUBLISHABLE_KEY') && defined('STRIPE_SECRET_KEY');
    if (!$checks['Stripe Keys Set']) {
        $warnings[] = 'Stripe API keys not configured in config.php';
    }
}

// Security checks
$checks['HTTPS Only (Production)'] = !defined('ENABLE_HTTPS_ONLY') || ENABLE_HTTPS_ONLY === false || 
    (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on');
if (!$checks['HTTPS Only (Production)'] && defined('ENABLE_HTTPS_ONLY') && ENABLE_HTTPS_ONLY) {
    $warnings[] = 'HTTPS is not enabled but ENABLE_HTTPS_ONLY is true';
}

$checks['Error Display Off (Production)'] = !defined('DISPLAY_ERRORS') || DISPLAY_ERRORS === false;
if (!$checks['Error Display Off (Production)']) {
    $warnings[] = 'DISPLAY_ERRORS should be false in production';
}

// Calculate overall status
$total_checks = count($checks);
$passed_checks = count(array_filter($checks));
$status_class = empty($errors) ? 'success' : 'error';
$status_text = empty($errors) ? 'All Critical Checks Passed!' : 'Installation Issues Found';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Gateway - Installation Verification</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 40px 20px;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 40px;
            color: white;
            text-align: center;
        }
        
        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }
        
        .status-banner {
            padding: 30px 40px;
            text-align: center;
            font-size: 20px;
            font-weight: 600;
        }
        
        .status-banner.success {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-banner.error {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .content {
            padding: 40px;
        }
        
        .section {
            margin-bottom: 30px;
        }
        
        .section h2 {
            font-size: 24px;
            margin-bottom: 20px;
            color: #1f2937;
        }
        
        .check-list {
            list-style: none;
        }
        
        .check-item {
            display: flex;
            align-items: center;
            padding: 12px;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .check-item:last-child {
            border-bottom: none;
        }
        
        .check-icon {
            width: 24px;
            height: 24px;
            margin-right: 12px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        
        .check-icon.pass {
            background: #10b981;
            color: white;
        }
        
        .check-icon.fail {
            background: #ef4444;
            color: white;
        }
        
        .alert {
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 16px;
            border-left: 4px solid;
        }
        
        .alert-error {
            background: #fee2e2;
            border-color: #ef4444;
            color: #991b1b;
        }
        
        .alert-warning {
            background: #fef3c7;
            border-color: #f59e0b;
            color: #92400e;
        }
        
        .alert h3 {
            font-size: 16px;
            margin-bottom: 8px;
        }
        
        .alert ul {
            margin-left: 20px;
        }
        
        .alert li {
            margin-bottom: 4px;
        }
        
        .progress {
            background: #e5e7eb;
            border-radius: 8px;
            height: 30px;
            overflow: hidden;
            margin-bottom: 10px;
        }
        
        .progress-bar {
            background: linear-gradient(135deg, #10b981, #059669);
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            transition: width 0.3s ease;
        }
        
        .next-steps {
            background: #f0f9ff;
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid #3b82f6;
        }
        
        .next-steps h3 {
            color: #1e40af;
            margin-bottom: 12px;
        }
        
        .next-steps ol {
            margin-left: 20px;
            color: #1e3a8a;
        }
        
        .next-steps li {
            margin-bottom: 8px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔐 Payment Gateway</h1>
            <p>Installation Verification</p>
        </div>
        
        <div class="status-banner <?php echo $status_class; ?>">
            <?php echo $status_text; ?>
        </div>
        
        <div class="content">
            <div class="section">
                <h2>Installation Progress</h2>
                <div class="progress">
                    <div class="progress-bar" style="width: <?php echo ($passed_checks / $total_checks) * 100; ?>%">
                        <?php echo $passed_checks; ?> / <?php echo $total_checks; ?> Checks Passed
                    </div>
                </div>
            </div>
            
            <?php if (!empty($errors)): ?>
            <div class="alert alert-error">
                <h3>❌ Critical Errors (Must Fix)</h3>
                <ul>
                    <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($warnings)): ?>
            <div class="alert alert-warning">
                <h3>⚠️ Warnings (Recommended to Fix)</h3>
                <ul>
                    <?php foreach ($warnings as $warning): ?>
                    <li><?php echo htmlspecialchars($warning); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <div class="section">
                <h2>System Checks</h2>
                <ul class="check-list">
                    <?php foreach ($checks as $check_name => $passed): ?>
                    <li class="check-item">
                        <div class="check-icon <?php echo $passed ? 'pass' : 'fail'; ?>">
                            <?php echo $passed ? '✓' : '✗'; ?>
                        </div>
                        <span><?php echo htmlspecialchars($check_name); ?></span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            
            <?php if (empty($errors)): ?>
            <div class="next-steps">
                <h3>✅ Next Steps</h3>
                <ol>
                    <li><strong>Delete this file</strong> (install-check.php) for security</li>
                    <li>Navigate to <code>/admin/login.php</code></li>
                    <li>Login with default credentials:
                        <ul>
                            <li>Username: <code>admin</code></li>
                            <li>Password: <code>Admin@123</code></li>
                        </ul>
                    </li>
                    <li><strong>Change the default password immediately!</strong></li>
                    <li>Configure your gateway settings in the admin panel</li>
                    <li>Create your first payment link and test it</li>
                    <li>Review the README.md for production deployment checklist</li>
                </ol>
            </div>
            <?php else: ?>
            <div class="next-steps" style="border-color: #ef4444; background: #fee2e2;">
                <h3 style="color: #991b1b;">🔧 Fix Required Issues</h3>
                <ol style="color: #7f1d1d;">
                    <li>Review the errors listed above</li>
                    <li>Follow the installation instructions in README.md</li>
                    <li>Import database.sql into your MySQL database</li>
                    <li>Configure config.php with correct database credentials</li>
                    <li>Ensure all required PHP extensions are installed</li>
                    <li>Refresh this page after making changes</li>
                </ol>
            </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
