<?php
require_once 'config.php';
require_once 'classes/Database.php';
require_once 'classes/Security.php';
require_once 'classes/PaymentGateway.php';

$linkId = $_GET['id'] ?? '';
$gateway = new PaymentGateway();

if (!$linkId) {
    die('Invalid payment link');
}

$paymentLink = $gateway->getPaymentLink($linkId);

if (!$paymentLink) {
    die('Payment link not found or expired');
}

$db = Database::getInstance();
$config = $db->fetchOne(
    "SELECT * FROM gateway_config WHERE id = ?",
    [$paymentLink['config_id']]
);

$publishableKey = $config['mode'] === 'live' ? 
    Security::decrypt($config['api_key']) : 
    STRIPE_PUBLISHABLE_KEY;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo Security::escape($paymentLink['title']); ?> - Secure Payment</title>
    <script src="https://js.stripe.com/v3/"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .payment-container {
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 500px;
            width: 100%;
            overflow: hidden;
        }
        
        .payment-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 30px;
            color: white;
            text-align: center;
        }
        
        .payment-header h1 {
            font-size: 24px;
            margin-bottom: 8px;
        }
        
        .payment-header p {
            opacity: 0.9;
            font-size: 14px;
        }
        
        .amount-display {
            background: rgba(255, 255, 255, 0.2);
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
            font-size: 36px;
            font-weight: 700;
        }
        
        .payment-form {
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }
        
        .form-input {
            width: 100%;
            padding: 14px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .form-input:focus {
            outline: none;
            border-color: #667eea;
        }
        
        #card-element {
            padding: 14px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            background: white;
        }
        
        #card-element.StripeElement--focus {
            border-color: #667eea;
        }
        
        #card-errors {
            color: #e74c3c;
            font-size: 13px;
            margin-top: 8px;
            min-height: 20px;
        }
        
        .btn-pay {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            transition: transform 0.2s;
            margin-top: 20px;
        }
        
        .btn-pay:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(102, 126, 234, 0.4);
        }
        
        .btn-pay:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .security-badges {
            display: flex;
            justify-content: center;
            gap: 15px;
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid #e0e0e0;
            font-size: 12px;
            color: #666;
        }
        
        .badge {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .spinner {
            display: none;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 0.6s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .alert {
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border: 1px solid #10b981;
        }
        
        .alert-error {
            background: #fee2e2;
            color: #991b1b;
            border: 1px solid #ef4444;
        }
        
        .payment-info {
            background: #f9fafb;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 13px;
            color: #666;
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <div class="payment-header">
            <h1><?php echo Security::escape($paymentLink['title']); ?></h1>
            <p><?php echo Security::escape($paymentLink['description']); ?></p>
            <div class="amount-display">
                <?php echo $paymentLink['currency']; ?> <?php echo number_format($paymentLink['amount'], 2); ?>
            </div>
        </div>
        
        <div class="payment-form">
            <div id="payment-message"></div>
            
            <div class="payment-info">
                <strong>🔒 Secure Payment</strong><br>
                Your payment information is encrypted and secure. We never store your card details.
            </div>
            
            <form id="payment-form">
                <div class="form-group">
                    <label for="customer-name">Full Name *</label>
                    <input type="text" id="customer-name" class="form-input" required placeholder="John Doe">
                </div>
                
                <div class="form-group">
                    <label for="customer-email">Email Address *</label>
                    <input type="email" id="customer-email" class="form-input" required placeholder="john@example.com">
                </div>
                
                <div class="form-group">
                    <label for="card-element">Card Information *</label>
                    <div id="card-element"></div>
                    <div id="card-errors"></div>
                </div>
                
                <button type="submit" id="submit-button" class="btn-pay">
                    <span id="button-text">Pay <?php echo $paymentLink['currency']; ?> <?php echo number_format($paymentLink['amount'], 2); ?></span>
                    <div class="spinner" id="spinner"></div>
                </button>
            </form>
            
            <div class="security-badges">
                <div class="badge">
                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path d="M8 0c-.69 0-1.843.265-2.928.56-1.11.3-2.229.655-2.887.87a1.54 1.54 0 0 0-1.044 1.262c-.596 4.477.787 7.795 2.465 9.99a11.777 11.777 0 0 0 2.517 2.453c.386.273.744.482 1.048.625.28.132.581.24.829.24s.548-.108.829-.24a7.159 7.159 0 0 0 1.048-.625 11.775 11.775 0 0 0 2.517-2.453c1.678-2.195 3.061-5.513 2.465-9.99a1.541 1.541 0 0 0-1.044-1.263 62.467 62.467 0 0 0-2.887-.87C9.843.266 8.69 0 8 0z"/>
                    </svg>
                    SSL Encrypted
                </div>
                <div class="badge">
                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                        <path d="m10.97 4.97-.02.022-3.473 4.425-2.093-2.094a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/>
                    </svg>
                    PCI Compliant
                </div>
                <div class="badge">
                    <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path d="M8 1a2 2 0 0 1 2 2v4H6V3a2 2 0 0 1 2-2zm3 6V3a3 3 0 0 0-6 0v4a2 2 0 0 0-2 2v5a2 2 0 0 0 2 2h6a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2z"/>
                    </svg>
                    Secure
                </div>
            </div>
        </div>
    </div>
    
    <script>
        const stripe = Stripe('<?php echo $publishableKey; ?>');
        const elements = stripe.elements();
        
        // Create card element
        const cardElement = elements.create('card', {
            style: {
                base: {
                    fontSize: '16px',
                    color: '#32325d',
                    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen, Ubuntu, sans-serif',
                    '::placeholder': {
                        color: '#aab7c4'
                    }
                },
                invalid: {
                    color: '#e74c3c'
                }
            }
        });
        
        cardElement.mount('#card-element');
        
        // Handle card errors
        cardElement.on('change', function(event) {
            const displayError = document.getElementById('card-errors');
            if (event.error) {
                displayError.textContent = event.error.message;
            } else {
                displayError.textContent = '';
            }
        });
        
        // Handle form submission
        const form = document.getElementById('payment-form');
        form.addEventListener('submit', async function(event) {
            event.preventDefault();
            
            const submitButton = document.getElementById('submit-button');
            const buttonText = document.getElementById('button-text');
            const spinner = document.getElementById('spinner');
            
            // Disable submit button
            submitButton.disabled = true;
            buttonText.style.display = 'none';
            spinner.style.display = 'block';
            
            try {
                // Create payment method
                const {error, paymentMethod} = await stripe.createPaymentMethod({
                    type: 'card',
                    card: cardElement,
                    billing_details: {
                        name: document.getElementById('customer-name').value,
                        email: document.getElementById('customer-email').value
                    }
                });
                
                if (error) {
                    throw new Error(error.message);
                }
                
                // Send to server
                const response = await fetch('process-payment.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        payment_method_id: paymentMethod.id,
                        link_id: '<?php echo $linkId; ?>',
                        customer_name: document.getElementById('customer-name').value,
                        customer_email: document.getElementById('customer-email').value
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Payment successful! Redirecting...', 'success');
                    setTimeout(() => {
                        if (result.redirect_url) {
                            window.location.href = result.redirect_url;
                        } else {
                            window.location.href = 'payment-success.php?id=' + result.transaction_id;
                        }
                    }, 2000);
                } else {
                    throw new Error(result.message || 'Payment failed');
                }
                
            } catch (err) {
                showMessage(err.message, 'error');
                submitButton.disabled = false;
                buttonText.style.display = 'inline';
                spinner.style.display = 'none';
            }
        });
        
        function showMessage(message, type) {
            const messageDiv = document.getElementById('payment-message');
            messageDiv.innerHTML = `<div class="alert alert-${type}">${message}</div>`;
            messageDiv.scrollIntoView({ behavior: 'smooth' });
        }
    </script>
</body>
</html>
