<?php
require_once 'config.php';
require_once 'classes/Database.php';
require_once 'classes/Security.php';
require_once 'classes/PaymentGateway.php';

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid request data']);
    exit;
}

// Validate required fields
$required = ['payment_method_id', 'link_id', 'customer_name', 'customer_email'];
foreach ($required as $field) {
    if (empty($data[$field])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Missing required field: {$field}"]);
        exit;
    }
}

// Rate limiting
if (!Security::checkRateLimit('payment_' . Security::getClientIP(), 10, 300)) {
    http_response_code(429);
    echo json_encode(['success' => false, 'message' => 'Too many payment attempts. Please try again later.']);
    exit;
}

try {
    $gateway = new PaymentGateway();
    
    // Sanitize customer data
    $customerData = [
        'name' => Security::sanitize($data['customer_name']),
        'email' => Security::sanitize($data['customer_email'])
    ];
    
    // Validate email
    if (!Security::validateEmail($customerData['email'])) {
        throw new Exception('Invalid email address');
    }
    
    // Process payment
    $result = $gateway->processPayment(
        $data['link_id'],
        $data['payment_method_id'],
        $customerData
    );
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Payment processed successfully',
        'transaction_id' => $result['transaction_id'],
        'status' => $result['status'],
        'redirect_url' => $result['redirect_url']
    ]);
    
} catch (Exception $e) {
    error_log("Payment Error: " . $e->getMessage());
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
