<?php
/**
 * Admin Password Reset Utility
 * 
 * This script allows you to reset admin passwords.
 * SECURITY: Delete this file after use!
 * 
 * Usage:
 * 1. Upload this file to your payment_gateway directory
 * 2. Visit it in your browser: http://yourdomain.com/payment_gateway/reset-admin-password.php
 * 3. Enter username and new password
 * 4. DELETE THIS FILE after resetting password
 */

require_once 'config.php';
require_once 'classes/Database.php';
require_once 'classes/Security.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $username = trim($_POST['username'] ?? '');
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        // Validation
        if (empty($username) || empty($new_password)) {
            throw new Exception('Username and password are required');
        }
        
        if ($new_password !== $confirm_password) {
            throw new Exception('Passwords do not match');
        }
        
        if (strlen($new_password) < 8) {
            throw new Exception('Password must be at least 8 characters long');
        }
        
        // Hash the new password
        $hashed_password = password_hash($new_password, PASSWORD_BCRYPT, ['cost' => 10]);
        
        // Update password in database
        $db = Database::getInstance();
        $result = $db->update(
            'admin_users',
            ['password' => $hashed_password],
            'username = ?',
            [$username]
        );
        
        // Check if user exists
        $user = $db->fetchOne("SELECT id FROM admin_users WHERE username = ?", [$username]);
        
        if (!$user) {
            throw new Exception("User '{$username}' not found");
        }
        
        $message = "Password successfully updated for user: {$username}";
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Admin Password - Payment Gateway</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            padding: 40px;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
            width: 100%;
            max-width: 500px;
        }
        
        h1 {
            color: #667eea;
            font-size: 28px;
            margin-bottom: 10px;
            text-align: center;
        }
        
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .warning {
            background: #fef3c7;
            border-left: 4px solid #f59e0b;
            padding: 15px;
            margin-bottom: 25px;
            border-radius: 5px;
            font-size: 13px;
            color: #92400e;
        }
        
        .warning strong {
            display: block;
            margin-bottom: 5px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 500;
            font-size: 14px;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 5px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .success-message {
            background: #d1fae5;
            color: #065f46;
            padding: 12px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 14px;
            border-left: 4px solid #10b981;
        }
        
        .error-message {
            background: #fee2e2;
            color: #991b1b;
            padding: 12px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 14px;
            border-left: 4px solid #ef4444;
        }
        
        .btn-reset {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .btn-reset:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .info-box {
            background: #f0f9ff;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            font-size: 13px;
            color: #075985;
        }
        
        .info-box strong {
            display: block;
            margin-bottom: 8px;
            color: #0369a1;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔐 Reset Admin Password</h1>
        <p class="subtitle">Payment Gateway Admin Utility</p>
        
        <div class="warning">
            <strong>⚠️ Security Warning</strong>
            DELETE this file immediately after resetting your password!
        </div>
        
        <?php if ($message): ?>
        <div class="success-message">
            <strong>✓ Success!</strong><br>
            <?php echo htmlspecialchars($message); ?>
            <br><br>
            <strong>Next steps:</strong>
            <ol style="margin-left: 20px; margin-top: 10px;">
                <li>DELETE this file (reset-admin-password.php)</li>
                <li>Go to <a href="admin/login.php" style="color: #065f46;">admin login page</a></li>
                <li>Login with your new password</li>
            </ol>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="error-message">
            <strong>✗ Error</strong><br>
            <?php echo htmlspecialchars($error); ?>
        </div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <div class="form-group">
                <label for="username">Admin Username</label>
                <input type="text" id="username" name="username" required 
                       placeholder="Enter username (default: Creatorschoice)">
            </div>
            
            <div class="form-group">
                <label for="new_password">New Password</label>
                <input type="password" id="new_password" name="new_password" required 
                       placeholder="Minimum 8 characters">
            </div>
            
            <div class="form-group">
                <label for="confirm_password">Confirm New Password</label>
                <input type="password" id="confirm_password" name="confirm_password" required 
                       placeholder="Re-enter new password">
            </div>
            
            <button type="submit" class="btn-reset">
                Reset Password
            </button>
        </form>
        
        <div class="info-box">
            <strong>Default Credentials:</strong>
            • Username: Creatorschoice<br>
            • Password: Creators@7620
            <br><br>
            If you forgot your username, check the <code>admin_users</code> table in your database.
        </div>
    </div>
</body>
</html>
